<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\Type;
use App\Models\Guide;

class GestionFrontController extends Controller
{
    public function index()
    {
        $guides = Guide::where('id_type', $tp)
                    ->orderBy('ordre', 'asc')
                    ->with('children')
                    ->paginate(10); 

        return view('backoffice.gestion_guide.listeguides', compact('type', 'guides'));
    }

    public static function getTypes()
    {
        return Type::orderBy('id_type', 'asc')->get();
    }


    public function getMenu()
    {
        $types = Type::with('guides.children') 
             ->orderBy('id_type')
             ->get();
        return view('layouts.header', compact('types'));
    }

    public function create()
    {

        $guides = Guide::where('id_type', $tp)
                        ->orderBy('ordre', 'asc')
                        ->get();

        $parentGuides = $this->buildGuideTree($guides);

        return view('backoffice.gestion_guide.create', compact('type', 'parentGuides'));
    }

    private function buildGuideTree($guides, $parentId = null, $prefix = '')
    {
        $tree = [];
        foreach ($guides->where('id_parent', $parentId) as $guide) {
            $tree[] = [
                'id_G' => $guide->id_G,
                'titre_G' => $prefix . $guide->titre_G
            ];
            $children = $this->buildGuideTree($guides, $guide->id_G, $prefix . '--');
            $tree = array_merge($tree, $children);
        }
        return $tree;
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'titre_G' => 'required|string|max:255',
            'titre_fr' => 'required|string|max:255',
            'titre_en' => 'required|string|max:255',
            'ordre' => 'required|integer',
            'type_contenu' => 'required|in:none,lien,builder',
            'id_type' => 'required|integer|exists:tp_type,id_type',
            'id_parent' => 'nullable|integer|exists:guides,id_G',
            'type' => 'required|in:menu_principal,sous_menu',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        Guide::create([
            'titre_G' => $request->titre_G,
            'titre_fr' => $request->titre_fr,
            'titre_en' => $request->titre_en,
            'type_contenu' => $request->type_contenu,
            'ordre' => $request->ordre,
            'type' => $request->type,
            'id_type' => $request->id_type,
            'id_parent' => $request->id_parent,
        ]);

        return redirect()->route('types.index', $request->id_type)->with('success', 'Guide ajouté avec succès!');
    }

    public function edit($id)
    {
        $guide = Guide::findOrFail($id);
        $type = Type::findOrFail($guide->id_type);
        $allGuides = Guide::where('id_type', $guide->id_type)
                        ->where('id_G', '!=', $guide->id_G)
                        ->orderBy('ordre', 'asc')
                        ->get();

        $parentGuides = $this->buildGuideTree($allGuides);

        return view('backoffice.gestion_guide.edit', compact('guide', 'type', 'parentGuides'));
    }

    public function update(Request $request, $id)
    {
        $guide = Guide::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'titre_G' => 'required|string|max:255',
            'titre_fr' => 'required|string|max:255',
            'titre_en' => 'required|string|max:255',
            'ordre' => 'required|integer',
            'type' => 'required|in:menu_principal,sous_menu',
            'type_contenu' => 'required|in:none,lien,builder',
            'id_type' => 'required|integer|exists:tp_type,id_type',
            'id_parent' => 'nullable|integer|exists:guides,id_G',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $guide->update([
            'titre_G' => $request->titre_G,
            'titre_fr' => $request->titre_fr,
            'titre_en' => $request->titre_en,
            'type_contenu' => $request->type_contenu,
            'ordre' => $request->ordre,
            'type' => $request->type,
            'id_type' => $request->id_type,
            'id_parent' => $request->id_parent,
        ]);

        return redirect()->route('guides.index', $guide->id_type)->with('success', 'Guide mis à jour avec succès !');
    }

    public function destroy($id)
    {
        $guide = Guide::findOrFail($id);

        if ($guide->children()->count() > 0) {
            return redirect()->back()->with('error', 'Impossible de supprimer un guide qui possède des sous-guides.');
        }

        $guide->delete();

        return redirect()->route('guides.index', $guide->id_type)->with('success', 'Guide supprimé avec succès !');
    }

}
